#!/bin/sh
#Troubelshooting script for Webcam Redirection

#*************************************************************************************
#*************************** Log Functions *******************************************
#*************************************************************************************
#Color Definition
INF="\033[1;33m"
ERR="\033[0;31m"
SUC="\033[0;32m"
EC="\033[0m"

disable_color(){
    INF=""
    ERR=""
    SUC=""
    EC=""
}

log_err() {
    log="$1"
    echo "${ERR}ERROR: ${log}${EC}"
}

log_info() {
    log="$1"
    echo "${INF}${log}${EC}"
}

log_succ() {
    log="$1"
    echo "${SUC}${log}${EC}"
}

#*************************************************************************************
#*************************** Helper Functions ****************************************
#*************************************************************************************
print_help(){

    echo "Usage: $0 [OPTION] [COLOR]" >&2
    echo "Checks if current system is ok to use Webcam Redirection and Rave. Default argument 
for OPTION will be --webcam."
    echo "Where:"
    echo "OPTION":
    echo "   -w, --webcam        Run diagnostic checks for webcam redirection."
    echo "   -r, --rave          Run diagnostic checks for rave."
    echo "   -h, --help          Display all allowed CLI arguments."
    echo "COLOR:"
    echo "   --no-color          Optional arg to turn off colors"

    exit 1
}

get_arch(){
    dpkg --print-architecture | sed "s/[^=]*=\([0-9]*\).*/\1/g"
}

get_gstreamer_path(){

    arch=$(get_arch)

    case "$arch" in
        amd64)
            echo "/usr/lib/x86_64-linux-gnu"
            ;;
        armhf)
            echo "/usr/lib/libarm-linux-gnueabihf"
            ;;
        arm)
            echo "/usr/lib/arm-linux-gnueabi"
            ;;
        arm64)
            echo "/usr/lib/aarch64-linux-gnu"
            ;;
    esac
}

get_install_path(){
    
    install_path=""
    gstreamer_path=$(get_gstreamer_path)

    plugin_path=""
    tail_path=""
    if [ -f "${gstreamer_path}/gstreamer-1.0/libgstflatstm.so" ]; then
        plugin_path=$(readlink -f "$gstreamer_path/gstreamer-1.0/libgstflatstm.so")
        tail_path="${plugin_path#/*/*/*/}"
    elif [ -f "${HOME}/.gstreamer-1.0/plugins/libgstflatstm.so" ]; then
        plugin_path=$(readlink -f "${HOME}/.gstreamer-1.0/plugins/libgstflatstm.so")
        tail_path="${plugin_path#/*/*/*/*/}"
    else
        plugin_path=""
    fi

    if [ -n "$plugin_path" ]; then
        install_path="${plugin_path%/$tail_path}"
    fi

    echo "$install_path"
}

get_config_file(){

    config_file=""
    install_path=$(get_install_path)

    if [ -f "${HOME}/.ICAClient/wfclient.ini" ]; then
        config_file="${HOME}/.ICAClient/wfclient.ini"
    elif [ -f "${install_path}/config/wfclient.template" ]; then
        config_file="${install_path}/config/wfclient.template"
    fi

    echo "$config_file"
}

read_option(){
    
    input="$1"
    config_option="$2"

    option_value=""
    while read -r line
    do
        case "$line" in
            "$config_option"*) 
                value=${line#*"="} #splitter
                if [ -n "$value" ] && [ "$value" != "*" ]; then
                    option_value="$value"
                fi
                break
                ;;
            *);;
        esac
    done < "$input"

    echo "$option_value"
}

check_bool_value(){
    value="$1"
    case "$value" in
        Yes)
            echo "True"
            ;;
         On)
            echo "True"
            ;;
       True)
            echo "True"
            ;;
          *)
            echo "False"
            break
            ;; 
    esac

}

list_encodecs(){
    h264_supp_encoder="vaapih264enc fluvah264enc fluh264enc openh264enc x264enc"
    result=255
    for i in $h264_supp_encoder; do
        gst-inspect-1.0 "$i" >/dev/null 2>&1
        if [ "$?" = "0" ]; then
            log_succ "$i found"
            result=0
        else 
            log_info "$i not found"
        fi
    done
    echo "\n" #separator
    return "$result"
}

get_h246_encoder(){
    h264_supp_encoder="vaapih264enc fluvah264enc fluh264enc openh264enc x264enc"
    encoder="not-found"
    for i in $h264_supp_encoder; do
        gst-inspect-1.0 "$i" >/dev/null 2>&1
        if [ "$?" = "0" ]; then
            encoder="$i" 
            break;
        fi
    done
    echo "$encoder"
}

#*************************************************************************************
#************************* System Dependencies ***************************************
#*************************************************************************************
system_deps(){

    gst_path=$(get_gstreamer_path)

    log_info "\n+++++++++++++++++++++++++++ System Dependencies  ++++++++++++++++++++++++++++++++++++\n"

    gstreamer_installed=1
    #************** Check gstreamer version
    log_info "GStreamer major version:"
    if [ -f "${gst_path}/libgstreamer-1.0.so.0" ]; then
        log_succ "GStreamer 1.0 installed.\n"
    elif [ -f "${gst_path}/libgstreamer-0.1.so.0" ]; then
        log_err "GStreamer 0.1 is not supported anymore for CWA Linux. Please install GStreamer 1.0 to continue.\n"
    else
        log_err "Please install GStreamer 1.0\n"
	gstreamer_installed=0
    fi

    #************** Check if gstreamer tools are ok
    log_info "Check GStreamer tools:"
    gst-inspect-1.0 --version >/dev/null 2>&1
    if [ "$?" -ne "0" ]; then
        log_err "gst-inspect-1.0 tool is not in system please install GStreamer 1.0."
	gstreamer_installed=0
    else
        log_succ "gst-inspect-1.0 tool installed."
    fi

    gst-launch-1.0 --version >/dev/null 2>&1
    if [ "$?" -ne "0" ]; then
        log_err "gst-launch-1.0 tool is not in system please install GStreamer 1.0.\n"
	gstreamer_installed=0
    else
        log_succ "gst-launch-1.0 tool installed.\n"
    fi

    if [ $gstreamer_installed -eq 0 ]; then
 	log_err "Some gstreamer deps are missed, to install them go to https://gstreamer.freedesktop.org/documentation/installing/on-linux.html?gi-language=c\n"
    else 
        #************** Listing all GStreamer in system
        log_succ "GStreamer installed version: $(gst-inspect-1.0 --gst-version)\n"

        #************** Check if any gstreamer plugins is blacklisted
        log_info "Listing GStreamer plugins in blacklist ...\n$(gst-inspect-1.0 -b)\n"
    fi
    
    log_info "Listing GStreamer installations ...\n\n$(dpkg -l | grep gstreamer)\n"

}


#*************************************************************************************
#****************** Basic Configuration for Webcam Redireciton ***********************
#*************************************************************************************
basic_config(){
    
    config_file="$1"

    log_info "\n++++++++++++++++++++ Basic Configuration for Webcam Redirection ++++++++++++++++++++\n"

    #************* Enable Helios
    config_option="AllowAudioInput"
    result=$(read_option "$config_file" "$config_option")
    log_info "${config_option}:"

    if [ $(check_bool_value "$result") = "True" ]; then
        log_succ "Webcam Redirection is enabled.\n"
    else
        log_err "Webcam Redirection is not enabled: please add \"AllowAudioInput=True\" under [WFClient]\
 section in ${config_file}.\n"
    fi

    #************* Enable Theora/H264 encodec
    config_option="HDXH264InputEnabled"
    result=$(read_option "$config_file" "$config_option")
    log_info "${config_option}:"

    if [ $(check_bool_value "$result") = "True" ]; then
        log_succ "H264 encoders are enabled.\n"
        log_info "Searching supported encoders..."
        list_encodecs
        if [ "$?" -ne "0" ]; then
            log_err "No supported H264 encoder found in the system, please install at least one of them.\n"
            log_info "Note: Since none of supported h264 encoder was found in system Webcam Redirection
will failback to Theora encoder, be aware that Theora only works with 32 bits app in VDA side.\n"
        fi
    else
        log_err "Since ${config_option} is disabled, Theora encodec is enabled by default.\n"

        #Check Quality property for Theora
        quality_option="HDXWebCamQuality"
        quality=$(read_option "$config_file" "$quality_option")
        log_info "Checking Quality for Theora encoder..."
        log_info "${quality_option}:"

        if [ -z "$quality" ]; then
            log_succ "${quality_option} is not set, so default quality of 48 is used for Theora encoder.\n"
            log_info "Note: Custom quality can be set using options \"${quality_option}\" under
[WFClient] section in ${config_file}, e.g. ${quality_option}=60.\n"
        else
            log_succ "Quality property for Theora encoder is set to ${quality}.\n"
        fi

        log_info "Note: Be aware that only 32 bits VDA's app can be used with Theora encodec, please add \"HDXH264InputEnabled=True\"
under [WFClient] section in ${config_file} file if you want to use our supported h264 encodecs.\n"
    fi

    #************* Webcam device 
    config_option="HDXWebCamDevice"
    result=$(read_option ${config_file} ${config_option})
    log_info "${config_option}:"

    if [ -z "$result" ]; then
        log_succ "Since ${config_option} is not set, default webcam /dev/video0 is used.\n"
    else
        log_succ "${config_option} is set to device: ${result}.\n"
    fi

}

#*************************************************************************************
#**************** Stream Configuration for Webcam Redirection ************************
#*************************************************************************************
stream_config(){
    
    install_path="$1"
    config_file="$2"

    log_info "\n+++++++++++++++++++ Stream Configuration for Webcam Redireciton ++++++++++++++++++++\n"

    #************* Resolution
    width_option="HDXWebCamWidth"
    height_option="HDXWebCamHeight"
    width=$(read_option "$config_file" "$width_option")
    height=$(read_option "$config_file" "$height_option")

    default_res="no"
    if [ -z "$width" ]; then
        width="640"
        default_res="yes"
    fi

    if [ -z "$height" ]; then
        height="480"
        default_res="yes"
    fi

    log_info "${width_option} and ${height_option}:"
    if [ "$default_res" = "yes" ]; then
        log_succ "Since ${width_option} or ${height_option} is not set, one of the deafult value is used, width=${width} and height=${height} is used.\n"
        log_info "Note: Custom resolution can be set using options \"${width_option}\" and \"${height_option}\" under
[WFClient] section in ${config_file}, e.g. ${width_option}=1920  ${height_option}=1080.\n"
    else
        log_succ "Resolution is set to ${width}x${height}.\n"
    fi

    #************* Frame Rate
    num_option="HDXWebCamFramesPerSec"
    den_option="HDXWebCamFramesPerSecDenominator"
    num=$(read_option "$config_file" "$num_option")
    den=$(read_option "$config_file" "$den_option")

    default_res="no"
    if [ -z "$num" ]; then
        num="30"
        default_res="yes"
    fi

    if [ -z "$den" ]; then
        den="1"
        default_res="yes"
    fi

    log_info "${num_option} and ${den_option}:"

    if [ "$default_res" = "yes" ]; then
        log_succ "Since ${num_option} or ${den_option} is not set, one of the default values is used, ${num}/${den} frame rate is used.\n"
        log_info "Note: Custom frame rate can be set using options \"${num_option}\" and \"${den_option}\" under
[WFClient] section in ${config_file}, e.g. ${num_option}=30  ${den_option}=1.\n"
    else
        log_succ "Frame rate is set to ${num}/${den}.\n"
    fi

    #***************** Debug Mode
    debug_option="HDXWebCamDebug"
    result=$(read_option "$config_file" "$debug_option")

    if [ $(check_bool_value "$result") = "True" ]; then
        log_info "HDXWebCamDebug:"
        log_info "Note: Since ${debug_option} is set, a file with the encoded frame will be created in /tmp folder,
please use this option only for troubleshooting porpouse.\n"
    fi

    #***************** Background Blur
    beffect_option="HDXWebCamEnableBackgndEffect"
    result=$(read_option "$config_file" "$beffect_option")
    log_info "${beffect_option}:"
    if [ $(check_bool_value "$result") = "True" ]; then
        log_succ "${beffect_option} is enabled.\n"  
        log_info "Checking Background Blur Effect in the system..."
        #check for correct installaion
        LD_LIBRARY_PATH="$install_path/lib" gst-inspect-1.0 ctxbeffect >/dev/null 2>&1
        if [ "$?" -ne "0" ]; then
            log_err "Background Blur Effect plugin was not found in system, please reinstall CWA or check if current version support this feature.\n"
        else 
            log_succ "Background Blur Effect Plugin is correctly installed in system.\n"
        fi
    else 
        log_info "${beffect_option} is not set. Check if your CWA version supports Background Blur Effect.\n"
    fi
}

#*************************************************************************************
#********************* Dynamic Checks for Webcam Redirection *************************
#*************************************************************************************
pipeline_check(){

    install_path="$1"
    config_file="$2"

    log_info "\n++++++++++++++++++++++ Dynamic Checks for Webcam Redirection +++++++++++++++++++++++\n"

    out_path="/tmp/helios"
    if [ -d "$out_path/" ]; then
        rm -r "$out_path"
    fi
    mkdir "$out_path"
    export GST_DEBUG=6
    export GST_DEBUG_NO_COLOR=1
    export GST_DEBUG_DUMP_DOT_DIR=${out_path}

    #*************Checking Webcam
    export GST_DEBUG_FILE="$out_path/webcam.log"
    config_option="HDXWebCamDevice"
    result=$(read_option ${config_file} ${config_option})
    if [ -z "$result" ]; then
        result="/dev/video0"
    fi
    log_info "Checking webcam mapped to ${result}..."
    gst-launch-1.0 v4l2src device="$result" num-buffers=5 ! videoconvert ! fakevideosink >/dev/null 2>&1
    if [ "$?" -ne "0" ]; then
        log_err "Something wrong starting webcam mapped to ${result}.\n"
        log_info "Note: Please change the webcam using option \"${config_option}=<mapped device>\" under [WFClient] 
section in ${config_file}.\n"
    else 
        log_succ "Webcam device is OK and ready to be used.\n"
    fi

    #***************Checking Background Blur
    export GST_DEBUG_FILE="$out_path/backgndblur.log"
    beffect_option="HDXWebCamEnableBackgndEffect"
    result=$(read_option "$config_file" "$beffect_option")
    if [ $(check_bool_value "$result") = "True" ]; then
        log_info "Checking Background Blur Effect plugin..."
        #check for correct installaion
        LD_LIBRARY_PATH="$install_path/lib" gst-inspect-1.0 ctxbeffect >/dev/null 2>&1
        if [ "$?" -ne "0" ]; then
            log_err "Background Blur Effect plugin was not found in system, please reinstall CWA or check if current version support this feature."
        else 
            LD_LIBRARY_PATH="${install_path}/lib" gst-launch-1.0 videotestsrc num-buffers=5 ! videoconvert ! ctxbeffect ! fakevideosink >/dev/null 2>&1
            if [ "$?" -ne "0" ]; then
                log_err "Something wrong trying Background Blur Effect Plugin.\n"
            else 
                log_succ "Background Blur Effect Plugin is OK and ready to be used.\n"
            fi
        fi
    fi

    #***************Checking Encoders
    export GST_DEBUG_FILE="$out_path/encoder.log"
    log_info "Checking current H264/Theora encoder..."

    config_option="HDXH264InputEnabled"
    result=$(read_option "$config_file" "$config_option")
    theora_check="no"
    encoder="not-found"
    if [ $(check_bool_value "$result") = "True" ]; then
        encoder=$(get_h246_encoder)
        if [ "$encoder" = "not-found" ]; then
            log_err "No supported H264 encoder found in the system, please install at least one of them\n"
            log_info "Since none of the supported H264 encoder was found in system Theora will be check:"
            theora_check="yes"
        else
            log_info "${encoder}:"
            LD_LIBRARY_PATH="${install_path}/lib" gst-launch-1.0 videotestsrc num-buffers=5 ! videoconvert ! "$encoder" ! fakesink >/dev/null 2>&1
            if [ "$?" -ne "0" ]; then
                log_err "Something wrong trying ${encoder} encoder.\n"
            else 
                log_succ "${encoder} encoder is OK and ready to be used.\n"
            fi
        fi
    else
        log_info "Since ${config_option} is disabled, Theora encoder is enabled by default, and it will be checked.\n"
        theora_check="yes"
    fi

    if [ "$theora_check" = "yes" ]; then
        log_info "Checking theoraenc encoder ..."
        LD_LIBRARY_PATH="${install_path}/lib" gst-launch-1.0 videotestsrc num-buffers=5 ! videoconvert ! theoraenc ! fakesink >/dev/null 2>&1
        if [ "$?" -ne "0" ]; then
            log_err "Something wrong trying theoraenc encoder.\n"
        else 
            log_succ "theoraenc encoder is OK and ready to be used.\n"
        fi
    fi

    log_info "Logs for dynamic checks can be found in ${out_path}"
    unset GST_DEBUG
    unset GST_DEBUG_FILE
    unset GST_DEBUG_NO_COLOR
    unset GST_DEBUG_DUMP_DOT_DIR

 }



#*************************************************************************************
#************************* Basic Configuration for Rave ******************************
#*************************************************************************************
basic_config_rave(){
    
    config_file="$1"

    log_info "\n+++++++++++++++++++++++++++ Basic Configuration for Rave +++++++++++++++++++++++++++\n"

    #************* Enable Rave
    config_option="SpeedScreenMMAVideoEnabled"
    video_enabled=$(read_option "$config_file" "$config_option")
    log_info "${config_option}:"

    if [ -z "$video_enabled" ]; then
      log_succ "${config_option} is not set, so Rave is enabled by default."
    else
      if [ $(check_bool_value "$video_enabled") = "False" ]; then
        log_err "${config_option} is set to False, so Rave is disabled, please set \"SpeedScreenMMAVideoEnabled=True\" under [WFClient] section in ${config_file} or remove the option completely, Rave is enabled by default.\n"
      else
        log_succ "${config_option} is set to True, so Rave is enabled. Notes this option can be removed completely since Rave is enabled by default.\n"
      fi
    fi


}

#*************************************************************************************
#************************* Stream Configuration for Rave *****************************
#*************************************************************************************
stream_config_rave(){
    
    install_path="$1"
    config_file="$2"

    log_info "\n++++++++++++++++++++++++++ Stream Configuration for Rave +++++++++++++++++++++++++++\n"

    #************* Overlay config
    config_option="SpeedScreenMMAStopOverlayHandlingEvents"
    overlay_disabled=$(read_option "$config_file" "$config_option")
    log_info "${config_option}:"

    if [ -z "$overlay_disabled" ]; then
      log_succ "${config_option} is not set, Overlay is enabled by default.\n"
    else
      if [ $(check_bool_value "$overlay_disabled") = "True" ]; then
        log_err "${config_option} is not set to False, so Overlay is disabled, please set \"${config_option}=False\" under [WFClient] section in ${config_file} or remove the option completely, Overlay is enabled by default.\n"
      else
        log_succ "${config_option} is set to False, so Overlay is enabled. Notes this option can be removed completely since Overlay is enabled by default.\n"
      fi
    fi

    #************* Buffer max size
    max_buffer_option="SpeedScreenMMAMaximumBufferSize"
    buffer_size=$(read_option "$config_file" "$max_buffer_option")

    default_buffer_size="30240"
    log_info "${max_buffer_option}:"
    if [ -z "$buffer_size" ]; then
        log_succ "Since ${max_buffer_option} is not set, the deafult value of ${buffer_size_default} KB is used.\n"
        log_info "Note: Custom max buffer size can be set using options \"${max_buffer_option}\" under
[WFClient] section in ${config_file}, e.g. ${max_buffer_option}=${buffer_size_default}. The default value is large enough in most of the cases.\n"
    else
        log_succ "Max buffer size is set to ${buffer_size}. Please note that a small max buffer sizes can impact the video quality, the default value ${buffer_size_default} is large enough in most of the cases.\n"
    fi

}

#*************************************************************************************
#*************************** Dynamic Checks for Rave *********************************
#*************************************************************************************
pipeline_check_rave(){

    install_path="$1"
    config_file="$2"

    log_info "\n+++++++++++++++++++++++++++++ Dynamic Checks for Rave ++++++++++++++++++++++++++++++\n"

    #*************Checking flatsm demux
    log_info "Checking flatstm demux..."
    gst-inspect-1.0 flatstm >/dev/null 2>&1

    if [ "$?" -ne "0" ]; then
        log_err "Flatstm demux is not in the system or it is blacklisted. Please resintall CWA in system.\n"
    else 
        log_succ "Flatstm demux filter is in the system and ready to use.\n"
    fi

    out_path="/tmp/rave"
    if [ -d "$out_path/" ]; then
        rm -r "$out_path"
    fi
    mkdir "$out_path"
    export GST_DEBUG=6
    export GST_DEBUG_NO_COLOR=1
    export GST_DEBUG_DUMP_DOT_DIR=${out_path}

    #*************Checking decodebin
    export GST_DEBUG_FILE="$out_path/rave.log"
    log_info "Listing all h264 decoders in the system:"
    log_succ "$(gst-inspect-1.0 | grep '264.*dec\|dec.*264')\n"

    log_info "Checking decodebin..."
    gst-launch-1.0 videotestsrc num-buffers=30 ! videoconvert ! x264enc !  decodebin ! videoconvert ! autovideosink >/dev/null 2>&1

    if [ "$?" -ne "0" ]; then
        log_err "Something wrong using decodebin.\n"
    else 
        log_succ "Decodebin is able to decode a h264 stream.\n"
    fi

    log_info "Logs for dynamic checks can be found in ${out_path}"
    unset GST_DEBUG
    unset GST_DEBUG_FILE
    unset GST_DEBUG_NO_COLOR
    unset GST_DEBUG_DUMP_DOT_DIR

 }



main(){
    
    #specify if script is running from a root user
    if [ "$(id -u)" != "0" ]; then
       log_info "Launched by regular user...\n"	
    else
       log_info "Launched by root user: please run as regular user or be sure your config files are in root specific files.\n"	
    fi

    #arguments
    opt="-w"
    color=""

    #number of arguments
    case "$#" in
        0)
            ;;
        1)
            opt="$1"
            if [ "$opt" = "--no-color" ]; then
                disable_color
            fi
            ;;
        2)
            opt="$1"
            color="$2"
            if [ "$color" != "--no-color" ]; then
                echo "Unknow parameters: "$color"\n"
                print_help
            else
                disable_color
            fi 
            ;;
        *)
            echo "Unknow parameters: $@\n"
            print_help
            ;;
    esac

    config_file=$(get_config_file)
    install_path=$(get_install_path)

    case $opt in
        -w | --webcam)
            system_deps
            basic_config ${config_file}
            stream_config ${install_path} ${config_file}
            pipeline_check ${install_path} ${config_file}
            ;;
        -r | --rave)
            system_deps
            basic_config_rave ${config_file}
            stream_config_rave ${install_path} ${config_file}
            pipeline_check_rave ${install_path} ${config_file}
            ;;
        -h | --help)
            print_help
            ;; 

            *)
            echo "Unknow parameters: $opt\n"
            print_help
            ;; 
    esac
}


#++++++++++++++++++main
main $@
